"use client"
import { useState } from "react"
import type React from "react"

import { useRouter } from "next/navigation"
import { SiteHeader } from "@/components/site-header"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"

export default function LoginPage() {
  const router = useRouter()
  const [isLogin, setIsLogin] = useState(true)
  const [email, setEmail] = useState("")
  const [password, setPassword] = useState("")
  const [username, setUsername] = useState("")
  const [error, setError] = useState("")
  const [loading, setLoading] = useState(false)

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setError("")
    setLoading(true)

    try {
      if (isLogin) {
        // Login logic
        const users = JSON.parse(localStorage.getItem("users") || "[]")
        const user = users.find((u: any) => u.email === email && u.password === password)

        if (user) {
          localStorage.setItem("currentUser", JSON.stringify(user))
          router.push("/dashboard")
        } else {
          setError("Invalid email or password")
        }
      } else {
        // Signup logic
        const users = JSON.parse(localStorage.getItem("users") || "[]")

        if (users.find((u: any) => u.email === email)) {
          setError("Email already exists")
          setLoading(false)
          return
        }

        const isFirstUser = users.length === 0

        const newUser = {
          id: Date.now().toString(),
          email,
          password,
          username,
          role: isFirstUser ? "admin" : "developer",
          verified: true, // Auto-verify all users for testing
          createdAt: new Date().toISOString(),
          permissions: {
            canPost: true,
            canEdit: true,
            canDelete: true,
          },
        }

        users.push(newUser)
        localStorage.setItem("users", JSON.stringify(users))
        localStorage.setItem("currentUser", JSON.stringify(newUser))

        router.push(isFirstUser ? "/admin" : "/dashboard")
      }
    } catch (err) {
      setError("An error occurred. Please try again.")
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="min-h-screen bg-background text-foreground relative overflow-x-hidden">
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <img
          src="/la-city-bg.jpg"
          alt=""
          className="w-full h-full object-cover"
          style={{ imageRendering: "high-quality" }}
        />
        <div className="absolute inset-0 bg-black/70"></div>
      </div>

      <SiteHeader />

      <main className="container mx-auto px-6 py-20 relative">
        <div className="max-w-md mx-auto">
          <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-8 space-y-6">
            <div className="space-y-2 text-center">
              <h1 className="text-3xl font-black text-white">{isLogin ? "Developer Login" : "Create Account"}</h1>
              <p className="text-gray-400">{isLogin ? "Sign in to manage your portfolio" : "Join as a developer"}</p>
            </div>

            <form onSubmit={handleSubmit} className="space-y-4">
              {!isLogin && (
                <div className="space-y-2">
                  <Label htmlFor="username" className="text-white">
                    Username
                  </Label>
                  <Input
                    id="username"
                    type="text"
                    value={username}
                    onChange={(e) => setUsername(e.target.value)}
                    required
                    className="bg-white/5 border-0 text-white focus-visible:ring-0 focus-visible:ring-offset-0"
                  />
                </div>
              )}

              <div className="space-y-2">
                <Label htmlFor="email" className="text-white">
                  Email
                </Label>
                <Input
                  id="email"
                  type="email"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  required
                  className="bg-white/5 border-0 text-white focus-visible:ring-0 focus-visible:ring-offset-0"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="password" className="text-white">
                  Password
                </Label>
                <Input
                  id="password"
                  type="password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  required
                  minLength={6}
                  className="bg-white/5 border-0 text-white focus-visible:ring-0 focus-visible:ring-offset-0"
                />
              </div>

              {error && (
                <div className="text-red-400 text-sm bg-red-500/10 border border-red-500/20 rounded p-3">{error}</div>
              )}

              <Button
                type="submit"
                disabled={loading}
                className="w-full bg-[#b7ff00] hover:bg-[#b7ff00]/90 text-black font-bold"
              >
                {loading ? "Loading..." : isLogin ? "Sign In" : "Create Account"}
              </Button>
            </form>

            <div className="text-center">
              <button
                type="button"
                onClick={() => {
                  setIsLogin(!isLogin)
                  setError("")
                }}
                className="text-white hover:underline text-sm"
              >
                {isLogin ? "Need an account? Sign up" : "Already have an account? Sign in"}
              </button>
            </div>
          </div>
        </div>
      </main>
    </div>
  )
}
